/*==================================================================================================
 *
 * log_transform2_integrated_ajd_c.c
 * 
 * Same as 'transform2_integrated_ajd_c', but calculate logarithm of transform
 *
 * Calculate the transform E(exp((u+i*v) * int_0^T X_t dt)) for u,v real numbers, where X is a basic
 * affine jump diffusion: dX_t = k*(theta-X_t)dt + sigma*sqrt(X_t)dB_t + dJ_t. The formula is taken
 * from Appendix A in "Credit Risk" by Duffie and Singleton (2003), even though their proof only
 * covers the Laplace transform, i.e. v=0, everything goes still through. For given AJD parameters,
 * the function evaluates the transform for column(!) vectors u and v of equal length and returns
 * two vectors with the real and imaginary part of the transform.
 *
 * The calling syntax is:
 *      [a,b] = log_transform2_integrated_ajd_c(u, v, x0, k, theta, sigma, L, mu, T)
 *
 * x0       ... initial value
 * k        ... speed of mean reversion
 * theta    ... mean reversion level
 * sigma    ... volatility parameter
 * L        ... jump intensity
 * mu       ... mean of exponentially distributed jump sizes
 * T        ... time horizon of interest
 *
 * Example Fourier tranform:
 *          [a,b] = log_transform2_integrated_ajd_c(0, 1, 0.01, 0.0022, 0.0656, 0.0415, 0.0180, 0.0607, 5)
 *          [a,b] = log_transform2_integrated_ajd_c(zeros(10000,1), (0.01:0.01:100)', 0.01, 0.0022, 0.0656, 0.0415, 0.0180, 0.0607, 5)
 * Example Laplace transform:
 *          [a,b] = log_transform2_integrated_ajd_c(-1, 0, 0.01, 1e4, 0.01, 1e-10, 0.001, 0.001, 5)
 *
 * Written by Andreas Eckner in February 2007. In case you find any bugs, please contact me at
 * Andreas@Eckner.com. Thanks!
 *================================================================================================*/

#include <math.h>
#include "matrix.h"
#include "mex.h"

#ifndef max
#define max(a,b) (((a) > (b)) ? (a) : (b))
#endif

/* Complex addition */
static double abs_double(double a)
{
    if (a > 0) {
        return a;
    } else {
        return -a;
    }
}

/* Complex addition */
static void add(double out_real[], double out_imag[], double a_real, double a_imag, double b_real, double b_imag)
{
    out_real[0] = a_real + b_real;
    out_imag[0] = a_imag + b_imag;
    return;
}

/* Complex multiplication */
static void mult(double out_real[], double out_imag[], double a_real, double a_imag, double b_real, double b_imag)
{
    out_real[0] = a_real*b_real - a_imag*b_imag;
    out_imag[0] = a_real*b_imag + a_imag*b_real;
    return;
}

/* Complex norm */
void norm(double out_real[], double out_imag[], double a_real, double a_imag)
{
    out_real[0] = sqrt(a_real*a_real + a_imag*a_imag);
    out_imag[0] = 0;
    return;
}

/* Complex 1/x */
static void inverse(double out_real[], double out_imag[], double a_real, double a_imag)
{
    double norm2_real, norm2_imag, tmp;
  
    norm(&norm2_real, &norm2_imag, a_real, a_imag);
    tmp = norm2_real * norm2_real;
    out_real[0] = a_real / tmp;
    out_imag[0] = -a_imag / tmp;
    return;
}

/* Complex a/b */
static void a_div_b(double out_real[], double out_imag[], double a_real, double a_imag, double b_real, double b_imag)
{
    double inv_real, inv_imag;
    
    inverse(&inv_real, &inv_imag, b_real, b_imag);
    mult(out_real, out_imag, a_real, a_imag, inv_real, inv_imag);
    return;
}

/* Complex log */
static void log_c(double out_real[], double out_imag[], double a_real, double a_imag)
{
    double r_real, r_imag, phi;
    
    norm(&r_real, &r_imag, a_real, a_imag);
    phi = acos(a_real / r_real);
    if (a_imag < 0) phi = -phi;
    /* if (a_imag < -3.14159265358979) phi = 2.0 * 3.14159265358979 + phi; */
    /* if (a_imag > 3.14159265358979) phi = -2.0 * 3.14159265358979 + phi; */
    
    out_real[0] = log(r_real);
    out_imag[0] = phi;
    return;
}

/* Complex exponent */
static void exp_c(double out_real[], double out_imag[], double a_real, double a_imag)
{
    double tmp;
    
    tmp = exp(a_real);
    out_real[0] = tmp * cos(a_imag);
    out_imag[0] = tmp * sin(a_imag);
    return;
}

/* Complex square root */
static void sqrt_c(double out_real[], double out_imag[], double a_real, double a_imag)
{
    double r_real, r_imag, phi, tmp;
    
    norm(&r_real, &r_imag, a_real, a_imag);
    phi = acos(a_real / r_real);
    if (a_imag < 0) phi = 4.0 * 3.14159265358979 - phi;
    tmp = sqrt(r_real);
     
    out_real[0] = tmp * cos(phi/2);
    out_imag[0] = tmp * sin(phi/2);
    return;
}

/* Recursive computation of loss distribution */
static void log_transform2_integrated_ajd_c(double out_real[], double out_imag[], double u_real[], double u_imag[],
                                        double x0[], double k[], double theta[], double sigma[], double L[], double mu[], double horizons[])
{
    double *gamma_real, *gamma_imag, *c1_real, *c1_imag, *d1_real, *d1_imag;
    double *b1_real, *b1_imag, *d2_real, *d2_imag, *a2_real, *a2_imag;
    double *c2_real, *c2_imag, *tmp_real, *tmp_imag, *alpha_real, *alpha_imag;
    double *beta_real, *beta_imag;
    
    double sigma2, *tmp2_real, *tmp2_imag, *tmp3_real, *tmp3_imag, *tmp4_real, *tmp4_imag, *tmp5_real, *tmp5_imag, *tmp6_real, *tmp6_imag;
    tmp2_real = malloc (sizeof (tmp2_real));
    tmp2_imag = malloc (sizeof (tmp2_imag));
    tmp3_real = malloc (sizeof (tmp3_real));
    tmp3_imag = malloc (sizeof (tmp3_imag));
    tmp4_real = malloc (sizeof (tmp4_real));
    tmp4_imag = malloc (sizeof (tmp4_imag));
    tmp5_real = malloc (sizeof (tmp5_real));
    tmp5_imag = malloc (sizeof (tmp5_imag));
    tmp6_real = malloc (sizeof (tmp6_real));
    tmp6_imag = malloc (sizeof (tmp6_imag));
    
    /* Test complex arithmetic */
    /* norm(out_real, out_imag, u_real[0], u_imag[0]); */
    /* log_c(out_real, out_imag, u_real[0], u_imag[0]); */
    /* exp_c(out_real, out_imag, u_real[0], u_imag[0]); */
    /* inverse(out_real, out_imag, u_real[0], u_imag[0]); */
    /* sqrt_c(out_real, out_imag, u_real[0], u_imag[0]); */
    /* mult(out_real, out_imag, u_real[0], u_imag[0], x0[0], k[0]); */
    /* a_div_b(out_real, out_imag, u_real[0], u_imag[0], x0[0], k[0]); */
    
    /* Calculate gamma */
    gamma_real = malloc (sizeof (gamma_real));
    gamma_imag = malloc (sizeof (gamma_imag));
    sigma2 = sigma[0] * sigma[0];
    mult(gamma_real, gamma_imag, -2*sigma2, 0, u_real[0], u_imag[0]);
    gamma_real[0] += k[0]*k[0];
    sqrt_c(gamma_real, gamma_imag, gamma_real[0], gamma_imag[0]);
    
    /* Calculate c1 */
    c1_real = malloc (sizeof (c1_real));
    c1_imag = malloc (sizeof (c1_imag));
    add(c1_real, c1_imag, k[0], 0, gamma_real[0], gamma_imag[0]);
    a_div_b(c1_real, c1_imag, c1_real[0], c1_imag[0], 2*u_real[0], 2*u_imag[0]);
    
    /* Calculate d1 */
    d1_real = malloc (sizeof (d1_real));
    d1_imag = malloc (sizeof (d1_imag));
    add(d1_real, d1_imag, -k[0], 0, gamma_real[0], gamma_imag[0]);
    a_div_b(d1_real, d1_imag, d1_real[0], d1_imag[0], 2*u_real[0], 2*u_imag[0]);
    
    /* Calculate b1 */
    b1_real = malloc (sizeof (b1_real));
    b1_imag = malloc (sizeof (b1_imag));
    mult(b1_real, b1_imag, 2*u_real[0], 2*u_imag[0], c1_real[0], c1_imag[0]);
    b1_real[0] -= k[0];
    mult(b1_real, b1_imag, d1_real[0], d1_imag[0], b1_real[0], b1_imag[0]);
    mult(tmp2_real, tmp2_imag, k[0], 0, c1_real[0], c1_imag[0]);
    tmp2_real[0] = tmp2_real[0] - sigma2;
    add(b1_real, b1_imag, b1_real[0], b1_imag[0], tmp2_real[0], tmp2_imag[0]);
    add(tmp2_real, tmp2_imag, -c1_real[0], -c1_imag[0], -d1_real[0], -d1_imag[0]);
    a_div_b(b1_real, b1_imag, b1_real[0], b1_imag[0], tmp2_real[0], tmp2_imag[0]);
    
    /* Calculate d2 */
    d2_real = malloc (sizeof (d2_real));
    d2_imag = malloc (sizeof (d2_imag));
    add(d2_real, d2_imag, d1_real[0], d1_imag[0], mu[0], 0);
    a_div_b(d2_real, d2_imag, d2_real[0], d2_imag[0], c1_real[0], c1_imag[0]);
    
    /* Calculate a2 */
    a2_real = malloc (sizeof (a2_real));
    a2_imag = malloc (sizeof (a2_imag));
    a_div_b(a2_real, a2_imag, d1_real[0], d1_imag[0], c1_real[0], c1_imag[0]);
    
    /* Calculate c2 */
    c2_real = malloc (sizeof (c2_real));
    c2_imag = malloc (sizeof (c2_imag));
    a_div_b(c2_real, c2_imag, -mu[0], 0, c1_real[0], c1_imag[0]);
    c2_real[0] += 1;
    
    /* Calculate tmp */
    tmp_real = malloc (sizeof (tmp_real));
    tmp_imag = malloc (sizeof (tmp_imag));
    mult(tmp_real, tmp_imag, b1_real[0], b1_imag[0], horizons[0], 0);
    exp_c(tmp_real, tmp_imag, tmp_real[0], tmp_imag[0]);
    
    /*  c1+d1*tmp */
    mult(tmp2_real, tmp2_imag, d1_real[0], d1_imag[0], tmp_real[0], tmp_imag[0]);
    add(tmp2_real, tmp2_imag, tmp2_real[0], tmp2_imag[0], c1_real[0], c1_imag[0]);
    
    /* b1*c1*d1 */
    mult(tmp3_real, tmp3_imag, b1_real[0], b1_imag[0], c1_real[0], c1_imag[0]); 
    mult(tmp3_real, tmp3_imag, tmp3_real[0], tmp3_imag[0], d1_real[0], d1_imag[0]);  
    
    /* Calculate first term of alpha */
    alpha_real = malloc (sizeof (alpha_real));
    alpha_imag = malloc (sizeof (alpha_imag));
    add(alpha_real, alpha_imag, -c1_real[0], -c1_imag[0], -d1_real[0], -d1_imag[0]);
    mult(alpha_real, alpha_imag, alpha_real[0], alpha_imag[0], k[0]*theta[0], 0);
    a_div_b(alpha_real, alpha_imag, alpha_real[0], alpha_imag[0], tmp3_real[0], tmp3_imag[0]);
    add(tmp4_real, tmp4_imag, c1_real[0], c1_imag[0], d1_real[0], d1_imag[0]);
    a_div_b(tmp4_real, tmp4_imag, tmp2_real[0], tmp2_imag[0], tmp4_real[0], tmp4_imag[0]);
    log_c(tmp4_real, tmp4_imag, tmp4_real[0], tmp4_imag[0]);
    mult(alpha_real, alpha_imag, alpha_real[0], alpha_imag[0], tmp4_real[0], tmp4_imag[0]);
    
    /* Calculate second term of alpha */
    a_div_b(tmp4_real, tmp4_imag, k[0]*theta[0]*horizons[0], 0, c1_real[0], c1_imag[0]);
    add(alpha_real, alpha_imag, alpha_real[0], alpha_imag[0],  tmp4_real[0], tmp4_imag[0]);
    
    /* Calculate third term of alpha */
    inverse(tmp4_real, tmp4_imag, c2_real[0], c2_imag[0]);
    mult(tmp4_real, tmp4_imag, tmp4_real[0]-1, tmp4_imag[0], L[0]*horizons[0], 0);
    add(alpha_real, alpha_imag, alpha_real[0], alpha_imag[0], tmp4_real[0], tmp4_imag[0]);
    
    /* b1*c2*d2 */
    mult(tmp5_real, tmp5_imag, b1_real[0], b1_imag[0], c2_real[0], c2_imag[0]); 
    mult(tmp5_real, tmp5_imag, tmp5_real[0], tmp5_imag[0], d2_real[0], d2_imag[0]);
    
    /* c2+d2*tmp */
    mult(tmp6_real, tmp6_imag, d2_real[0], d2_imag[0], tmp_real[0], tmp_imag[0]);
    add(tmp6_real, tmp6_imag, tmp6_real[0], tmp6_imag[0], c2_real[0], c2_imag[0]);
    
    /* Calculate fourth term of alpha */
    mult(tmp4_real, tmp4_imag, a2_real[0], a2_imag[0], c2_real[0], c2_imag[0]);
    add(tmp4_real, tmp4_imag, tmp4_real[0], tmp4_imag[0], -d2_real[0], -d2_imag[0]);
    mult(tmp4_real, tmp4_imag, tmp4_real[0], tmp4_imag[0], L[0], 0);
    a_div_b(tmp4_real, tmp4_imag, tmp4_real[0], tmp4_imag[0], tmp5_real[0], tmp5_imag[0]);
    add(tmp3_real, tmp3_imag, c2_real[0], c2_imag[0], d2_real[0], d2_imag[0]);
    a_div_b(tmp3_real, tmp3_imag, tmp6_real[0], tmp6_imag[0], tmp3_real[0], tmp3_imag[0]);
    log_c(tmp3_real, tmp3_imag, tmp3_real[0], tmp3_imag[0]);
    mult(tmp4_real, tmp4_imag, tmp4_real[0], tmp4_imag[0], tmp3_real[0], tmp3_imag[0]);
    add(alpha_real, alpha_imag, alpha_real[0], alpha_imag[0], tmp4_real[0], tmp4_imag[0]);
    
    /* Calculate beta */
    beta_real = malloc (sizeof (beta_real));
    beta_imag = malloc (sizeof (beta_imag));
    add(beta_real, beta_imag, 1, 0, -tmp_real[0], -tmp_imag[0]);
    a_div_b(beta_real, beta_imag, beta_real[0], beta_imag[0], tmp2_real[0], tmp2_imag[0]);
    
    /* Calculate log-transform */
    mult(out_real, out_imag, beta_real[0], beta_imag[0], x0[0], 0);
    add(out_real, out_imag, out_real[0], out_imag[0], alpha_real[0], alpha_imag[0]);
    /* exp_c(out_real, out_imag, out_real[0], out_imag[0]); */
    
    /* Free up memory again */
    free(tmp_real); free(tmp_imag);
    free(tmp2_real); free(tmp2_imag);
    free(tmp3_real); free(tmp3_imag);
    free(tmp4_real); free(tmp4_imag);
    free(tmp5_real); free(tmp5_imag);
    free(tmp6_real); free(tmp6_imag);
    free(gamma_real); free(gamma_imag);
    free(c1_real); free(c1_imag);
    free(d1_real); free(d1_imag);
    free(b1_real); free(b1_imag);
    free(d2_real); free(d2_imag);
    free(a2_real); free(a2_imag);
    free(c2_real); free(c2_imag);
    free(alpha_real); free(alpha_imag);
    free(beta_real); free(beta_imag);
    return;
}

/* Gateway routine (to Matlab) */
void mexFunction( int nlhs, mxArray *plhs[], 
                  int nrhs, const mxArray*prhs[] )
     
{ 
    double *x0,  *k, *theta, *sigma, *L, *mu, *horizons;
    double *u_real, *u_imag, *out_real, *out_imag;
    unsigned int num_points, i;
   
    /* Assign pointers to the input variables parameters */ 
    u_real = mxGetPr(prhs[0]);
    u_imag = mxGetPr(prhs[1]);
    x0 = mxGetPr(prhs[2]);
    k = mxGetPr(prhs[3]);
    theta = mxGetPr(prhs[4]);
    sigma =mxGetPr(prhs[5]);
    L = mxGetPr(prhs[6]);
    mu = mxGetPr(prhs[7]);
    horizons = mxGetPr(prhs[8]);
    
    /* Create a matrix for the return argument */ 
    num_points = mxGetM(prhs[0]);
    plhs[0] = mxCreateDoubleMatrix(num_points, 1, mxREAL);
    plhs[1] = mxCreateDoubleMatrix(num_points, 1, mxREAL);
    out_real = mxGetPr(plhs[0]);
    out_imag = mxGetPr(plhs[1]);
    
    /* Do the actual computations in a subroutine */
    for (i=0; i<num_points; i++)
    {
        /* Test if input equal to zero */
        if ((abs_double(u_real[i]) < 1e-10) & (abs_double(u_imag[i]) < 1e-10))
        {
            out_real[i] = 0;
            out_imag[i] = 0;
        } else {
            log_transform2_integrated_ajd_c(&(out_real[i]), &(out_imag[i]), &(u_real[i]), &(u_imag[i]), x0, k, theta, sigma, L, mu, horizons);
        }
    }
    /* out_real[0] = u_real[0]; */
    /* out_imag[0] = u_imag[0]; */
    return;
}
